﻿#include  "StdAfx.h"

#include  <szPath.hpp>
#include  <szRuntimeException.hpp>
#include  <buffers.hpp>
#include  <tchar.h>

using namespace std;

SZ_AN_BEG

const szchar DIR_SEP = SZL('\\');
const szchar EXT_SEP = SZL('.');
const szchar DRV_SEP = SZL(':');
const szchar QUALIFY = SZL('?');

SZ_AN_END

SZ_NS_BEG(szpp)

SZ_SPEC
PathType GetPathType(const szstring &path)
{
  if (!path.empty())
  {
    if (1 == path.size())
      return DIR_SEP == path[0] ? PATH_TYPE_ROOT : PATH_TYPE_RELATIVE;
    else
    {
      const szchar first = path[0], second = path[1];
      if (DIR_SEP == first)
      {
        if (DIR_SEP == second)
        {
          if (path.size() > 2 && DIR_SEP != path[2])
            return QUALIFY == path[2] ? PATH_TYPE_QUALIFIED : PATH_TYPE_UNC;
        }
        else
          return PATH_TYPE_ROOT;
      }
      else
      {
        if (_istalpha(first) && DRV_SEP == second)
          return PATH_TYPE_DRIVE;
        else
          return PATH_TYPE_RELATIVE;
      }
    }
  }
  return PATH_TYPE_INVALID;
}

SZ_SPEC
szstring ExtractFileName(const szstring &path)
{
  const szstring::size_type pos = path.find_last_of(DIR_SEP);
  return (szstring::npos != pos) ? path.substr(pos + 1) : path;
}

SZ_SPEC
szstring ExtractStem(const szstring &path)
{
  szstring::size_type       ext_pos = path.find_last_of(EXT_SEP);
  const szstring::size_type dir_pos = path.find_last_of(DIR_SEP);
  if (dir_pos + 1 == ext_pos)
    ext_pos = szstring::npos; // dir\.htaccess みたいなものはピリオドなしと考える

  if (szstring::npos == dir_pos) // ディレクトリセパレータがないので先頭からピリオド直前まで返す
    return path.substr(0, ext_pos);
  else // ディレクトリセパレータがあるのでその次からピリオド直前（ピリオドがないなら末尾）まで返す    
    return path.substr(dir_pos + 1, (dir_pos < ext_pos ? ext_pos - (dir_pos + 1) : szstring::npos));
}

SZ_SPEC
szstring ExtractExtension(const szstring &path, bool inclusive)
{
  szstring::size_type pos = path.find_last_of(EXT_SEP);
  const szstring::size_type dir_pos = path.find_last_of(DIR_SEP);
  if (dir_pos != szstring::npos && pos < dir_pos)
    pos = szstring::npos;
  return (szstring::npos != pos) ? path.substr(pos + (inclusive ? 0 : 1)) : SZL_EMPTY;
}

SZ_SPEC
szstring ExtractDirectory(const szstring &path, bool inclusive)
{
  const szstring::size_type pos = path.find_last_of(DIR_SEP);
  return (szstring::npos != pos) ? path.substr(0, pos + (inclusive ? 1 : 0)) : SZL_EMPTY;
}

SZ_SPEC
szstring RemoveExtension(const szstring &path)
{
  szstring::size_type pos = path.find_last_of(EXT_SEP);
  const szstring::size_type dir_pos = path.find_last_of(DIR_SEP);
  if (dir_pos != szstring::npos && pos < dir_pos)
    pos = szstring::npos;
  return path.substr(0, pos);
}

SZ_SPEC
szstring ChangeFileName(const szstring &path, const szstring &newName)
{
  return Combine(ExtractDirectory(path), newName);
}

SZ_SPEC
szstring ChangeExtension(const szstring &path, const szstring &ext)
{
  szstring retval = RemoveExtension(path);
  retval.append(1, EXT_SEP);
  retval.append(ext);
  return retval;
}

SZ_SPEC
szstring Combine(const szstring &path1, const szstring &path2, bool path1EndsWithSep)
{
  szstring retval(path1);
  retval.reserve(path1.size() + path2.size() + !path1EndsWithSep);
  if (!path1EndsWithSep)
    retval.append(1, DIR_SEP);
  retval.append(path2);
  return retval;
}

SZ_SPEC
void TrimSeparator(szstring *path)
{
  szstring::size_type searchPos = path->find_last_not_of(DIR_SEP);
  if (searchPos < path->size() - 1)
    path->erase(searchPos + 1); // 末尾にバックスラッシュが（複数）あれば削除
}

SZ_SPEC
std::deque<szstring> Decompose(const szstring &path)
{
  std::deque<szstring> q;
  szstring::size_type head = 0;
  for (szstring::size_type i = 0; i < path.size(); ++i)
  {
    if (DIR_SEP == path[i])
    {
      if (i > head)
        q.push_back(path.substr(head, i - head));
      head = i + 1;
    }
  }
  if (!path.empty() && head < path.size())
    q.push_back(path.substr(head));
  return q;
}

SZ_SPEC
szstring Compose(const std::deque<szstring> &elements)
{
  szstring path;
  for (std::deque<szstring>::const_iterator it = elements.begin(); it != elements.end(); ++it)
  {
    if (it != elements.begin())
      path.append(1, DIR_SEP);
    path.append(*it);
  }
  return path;
}

SZ_SPEC
szstring ConvertToLongPath(const szstring &path)
{
  sbuf<szchar, MAX_PATH> buf;
  if (0 == GetLongPathName(path.c_str(), (szchar *)buf, buf.size()))
    return path; // ファイル／フォルダが存在しない場合はエラーが起きるので、そのときはオリジナルのパスをそのまま返す。
  return szstring((const szchar *)buf);
}

SZ_SPEC
szstring Normalize(const szstring &path)
{
  //  src (deque) に分解したパス要素を格納し、先頭の要素から一つずつ dst (deque) にコピーする。
  //  その際 ".." という要素が見つかれば dst の末尾の要素をポップすることで、上位ディレクトリ
  //  への移動を実現する。絶対パスの場合には、パス文字列の先頭に必ず残さなければならない要素が
  //  存在するので、それらは prefix に格納しておく。

  deque<szstring> src = Decompose(path), dst;

  if (src.empty())
    return path;

  szstring prefix;
  const PathType t = GetPathType(path);
  switch (t)
  {
  case PATH_TYPE_UNC:
    prefix.assign(3, DIR_SEP);
    prefix.insert(2, src.front());
    src.pop_front();
    break;
  case PATH_TYPE_DRIVE:
    prefix = src.front();
    prefix.append(1, DIR_SEP);
    src.pop_front();
    break;
  case PATH_TYPE_ROOT:
    prefix.assign(1, DIR_SEP);
    break;
  }

  for (deque<szstring>::const_iterator it = src.begin(); it != src.end(); ++it)
  {
    if (0 == it->compare(SZL("..")))
    {
      if (!dst.empty())
        dst.pop_back();
      else if (t == PATH_TYPE_RELATIVE)
        dst.push_back(*it);
    }
    else if (0 != it->compare(SZL(".")))
      dst.push_back(*it);
  }

  if (!prefix.empty())
    dst.push_front(prefix);

  return Compose(dst);
}

SZ_SPEC
szstring CreateTempFileName(const szstring &dirPath)
{
  sbuf<szchar, MAX_PATH + 1> buf;
  if (0 == GetTempFileName(dirPath.c_str(), SZL("EPOTMP"), 0, (szchar *)buf))
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Cannot create temporary file")) << last_error_info(GetLastError()));
  return szstring((const szchar *)buf);
}

SZ_NS_END(szpp)

